//******************************************************************************
//
// Gadgeteer: Panel.cpp
//
// Handles the display of the panel and its controls
//
// 2016/4/18 -- Copyright 2016 Peter Goodeve
//
//******************************************************************************

#include <stdio.h>
#include <Application.h>
#include <Entry.h>
#include <Path.h>
#include <String.h>
#include <Button.h>
#include <StringView.h>
#include <TextControl.h>
#include <Menu.h>
#include <PopUpMenu.h>
#include <MenuItem.h>
#include <MenuBar.h>
#include <Button.h>
#include <CheckBox.h>

#include "WMPanel.h"
#include "WMSlider.h"
#include "Element.h"


class GTextControl : public BTextControl {
public:
	GTextControl(BRect frame, const char* name,
				const char* label, const char* initialText,
				BMessage* message,
				uint32 resizeMode = B_FOLLOW_LEFT | B_FOLLOW_TOP,
				uint32 flags = B_WILL_DRAW | B_NAVIGABLE)
				: BTextControl(frame, name, label, initialText,
							message, resizeMode, flags) {}
	void MessageReceived(BMessage *msg);
};

void GTextControl::MessageReceived(BMessage *msg) {
	if (msg->WasDropped() && msg->what == 'DATA') {
		entry_ref r;
		msg->FindRef("refs", &r);	// only get the first one
		BPath path = BPath(&r);
		SetText(path.Path());
	}
	else BTextControl::MessageReceived(msg);

}


class Panel : public WMPanel {
public:
	Panel(BPoint pos, PanelElem *pelem);
	void MessageReceived(BMessage *msg);
	bool QuitRequested();
	bool OutputData();
};

Panel::Panel(BPoint pos, PanelElem *pelem)
 : WMPanel(pos, (WMParam *)pelem, "Control Panel")
{
	SetMargins(2, 2);
	SetPadding(2, 2);
	
	if (pelem->label) SetTitle(pelem->label);

	BRect frame;
	Element **controls = (Element **)pelem->controls.Items();
	int32 ncontrols = pelem->controls.CountItems();
	for (int i = 0; i < ncontrols; i++) {
		Element *ctrlelem = controls[i];
		switch (ctrlelem->posmode) {
		case RIGHT:
			frame = NextFrameRight(ctrlelem->width, ctrlelem->height);
			break;
		case BELOW:
			frame = NextFrameBelow(ctrlelem->width, ctrlelem->height);
			break;
		case RIGHT_TOP:
			frame = NextRightAtTop(ctrlelem->width, ctrlelem->height);
			break;
		case BELOW_LEFT:
			frame = NextBelowAtLeft(ctrlelem->width, ctrlelem->height);
			break;
		}
		/// ... add offsets here
		frame.OffsetBy(ctrlelem->xoffset, ctrlelem->yoffset);
		BView *control = NULL;	// Lowest Common Denominator!
		switch (ctrlelem->eltype) {
		case TEXTCTRL_ELEM:
			{
				char *label = ctrlelem->label[0] ? ctrlelem->label : NULL;
				control = new GTextControl(frame, "txtctrl", label,
					 ((TextCtrlElem *)ctrlelem)->initstr, new BMessage('ctrl'));
					 // (Message not used currently)
				ctrlelem->widget = control;
				((GTextControl *)control)->SetDivider(((TextCtrlElem *)ctrlelem)->divider);
			}
			break;

		case MENU_ELEM:
			{
				BMenuItem * b;
				BMenuBar * mbar
					= new BMenuBar(frame, B_EMPTY_STRING, 0);
				AddChild(mbar);
				BPopUpMenu *menu = new BPopUpMenu("");
				mbar->AddItem(menu);
	
				char *tag;
				MenuElem *melem = (MenuElem *)ctrlelem;
				for (int i=0; tag = (char *)melem->items.ItemAt(i); i++) {
					b = new BMenuItem(tag, new BMessage('menu'));
					menu->AddItem(b);
					if (i == melem->marked || i == 0) b->SetMarked(true);	// one of them has to be...
				}
				ctrlelem->widget = menu;
				SetLastFrame(frame);	// eek! menu doesn't respect frame!
			}
			break;

		case LABEL_ELEM:
			control = new BStringView(frame, "label", ctrlelem->label);
			// OTOH, a label resizes its frame to it's contents!
			break;

		case BUTTON_ELEM:
			{
				BMessage *msg = new BMessage('usbt');
				msg->AddPointer("widget", ctrlelem);
				control = new BButton(frame, "button", ctrlelem->label, msg);
				ctrlelem->widget = control;
			}
			break;
		case CHECKBOX_ELEM:
			{
				BCheckBox * ckbox = new BCheckBox(frame, "checkbox",
						 ctrlelem->label, NULL);
				ckbox->SetValue(((CheckBoxElem *)ctrlelem)->state);
				ctrlelem->widget = control = ckbox;
			}
			break;
		case SLIDER_ELEM:
			{
				SliderElem *selem = (SliderElem *)ctrlelem;
				WMSlider * sldr = new WMSlider(frame, selem->min, selem->max, NULL,
					 ctrlelem->label, NULL);
				sldr->SetValue(selem->init);
				ctrlelem->widget = control = sldr;
			}
			break;
		}
		if (control) AddChild(control);
	}
	
	SetLastFrame(MarginRect());	//Ensure buttons below everything else

	frame = NextBelowAtLeft(50, 18);
	if (pelem->cancel) {	// suppressed by null label
		BButton *cancelBtn = new BButton(frame,
								"Cncl button", pelem->cancel,
								new BMessage('cnbt'));
		AddChild(cancelBtn);
		frame = NextFrameRight(50, 18).OffsetBySelf(20, 0);
	}
	if (pelem->done) {	// suppressed by null label
		BButton *doneBtn = new BButton(frame,
									"OK button", pelem->done,
									new BMessage('okbt'));
		AddChild(doneBtn);
	}
}


void Panel::MessageReceived(BMessage *msg)
{
	switch(msg->what) {
	case 'cnbt':
		this->PostMessage(B_QUIT_REQUESTED);
		break;
	case 'okbt':
		OutputData();
		this->PostMessage(B_QUIT_REQUESTED);
		break;
	case 'usbt':
		{
			ButtonElem *elem = NULL;
			msg->FindPointer("widget", (void **)&elem);
			elem->state = true;
			OutputData();
			elem->state = false;
			// but don't quit unless told to...
			if (elem->close)
				this->PostMessage(B_QUIT_REQUESTED);
		}
		break;
	case 'ctrl':
		break;
	default:
		BWindow::MessageReceived(msg);
		break;
	}
}

bool Panel::QuitRequested()
{
	be_app->PostMessage(B_QUIT_REQUESTED);
	return true;
}


bool Panel::OutputData()
{
	BList widglist = ((PanelElem *)Param())->controls;
	Element *elem;
	for (int i=0; elem = (Element *)widglist.ItemAt(i); i++) {
		switch (elem->eltype) {
		case TEXTCTRL_ELEM:
			{
				GTextControl * tc = (GTextControl *)elem->widget;
				printf("%s\n", tc->Text());
			}
			break;
		case MENU_ELEM:
			{
				BPopUpMenu *menu = (BPopUpMenu *)elem->widget;
				BMenuItem *item = menu->FindMarked();
				printf("%s\n", item->Label());
			}
			break;
		case BUTTON_ELEM:
			{
				BButton *button = (BButton *)elem->widget;
				printf("%s\n", ((ButtonElem *)elem)->state ? button->Label() : "");
			}
			break;
		case CHECKBOX_ELEM:
			{
				BCheckBox *chkbox = (BCheckBox *)elem->widget;
				printf("%s\n", chkbox->Value() ? chkbox->Label() : "");
			}
			break;
		case SLIDER_ELEM:
			{
				WMSlider *sldr = (WMSlider *)elem->widget;
				printf("%d\n", sldr->Value());
			}
			break;
		}
	}
	fflush(NULL);
}


int buildPanel(PanelElem *spec) {
	Panel *panel = new Panel(spec->panelPos, spec);
	panel->Show();
}
