//******************************************************************************
//
// Gadgeteer: ReadYaml.cpp
//
// Reads the command-line arguments as a YAML string
// and builds the panel and its controls
//
// 2016/6/4 -- Copyright 2016 Peter Goodeve
//
//******************************************************************************


#include <stdlib.h>
#include <stdio.h>
#include <String.h>

#include "yaml.h"
#include "WMPanel.h"
#include "Element.h"

char * Event_Types[] = {
	"YAML_NO_EVENT",

	"YAML_STREAM_START_EVENT",
	"YAML_STREAM_END_EVENT",

	"YAML_DOCUMENT_START_EVENT",
	"YAML_DOCUMENT_END_EVENT",

	"YAML_ALIAS_EVENT",
	"YAML_SCALAR_EVENT",

	"YAML_SEQUENCE_START_EVENT",
	"YAML_SEQUENCE_END_EVENT",

	"YAML_MAPPING_START_EVENT",
	"YAML_MAPPING_END_EVENT"
};

char * Keys[] = {
	"panel",
	"title",
	"cancel",
	"done",
	"width",
	"height",
	"pos",
	"label",
	"init",
	"div",
	"min",
	"max",
	"items",
	"textcontrol",
	"menu",
	"button",
	"checkbox",
	"slider",
	NULL
};

KeyID getKey(unsigned char *keystr) {
	char *key;
	int32 id;
	for (id=0; key=Keys[id]; id++) {
		if (strcmp((char *)keystr, key) == 0) break;
	}
	return (KeyID)id; 
}


Element::Element(ElemType etype, Element *Parent=NULL) : parent(Parent), currmode(NONE), prevmode(NONE), posmode(DEFAULT),
		xoffset(0), yoffset(0), width(0), height(0), label(NULL), eltype(etype) {
	}
	bool Element::getPos(char *pstr) {
		char wstr[128];	// should be enough...
		char *token[5] ,*strp = wstr;
		int32 indx = 0;
		strncpy(wstr, pstr, 128);
		wstr[127] = 0;
		while (token[indx] = strtok(strp, " ")) {
			strp = NULL;
			if (indx < 4) indx++;	// strtok is WEIRD! -- must complete!
			
		}
		if (!indx) return false;
		int32 toki = 0;
		if (strcmp(token[toki], "right") == 0) {
			if (++toki < indx && strcmp(token[toki], "top") == 0) {
				posmode = RIGHT_TOP;
				toki++;
			}
			else posmode = RIGHT;
		}
		else if (strcmp(token[toki], "below") == 0) {
			if (++toki < indx && strcmp(token[toki], "left") == 0) {
				posmode = BELOW_LEFT;
				toki++;
			}
			else posmode = BELOW;
		}
		int32 val;
		char *endptr;
		if (toki < indx) {
			val = strtol(token[toki], &endptr, 10);
			if (endptr > (char *)token[toki]) xoffset = val;
			toki++;
		}
		if (toki < indx) {
			val = strtol(token[toki], &endptr, 10);
			if (endptr > (char *)token[toki]) yoffset = val;
		}
		return true;
	}


bool Element::AddPair(unsigned char *key, unsigned char *value) {
	char *valstr = (char *)value;	// annoyance avoidance....
	switch(getKey(key)) {
	case WIDTH:
		width = atoi(valstr);
		break;
	case HEIGHT:
		height = atoi(valstr);
		break;
	case LABEL:
		label = valstr;
		break;
	case POS:
		getPos(valstr);
		break;
	default:
		return false;
	}
	return true;
}

Element *Element::AddKey(unsigned char *key) {return NULL;}


PanelElem::PanelElem() : Element(PANEL)
{
	width=100;
	height=20;
	posmode=BELOW;
	cancel = "CANCEL";
	done = "DONE";
	panelPos = BPoint(100,100);
	divider =-1; // ignored initially
	min = 0; max = 100;	// default slider limits
}

bool PanelElem::AddPair(unsigned char *key, unsigned char *value) {
	char *valstr = (char *)value;	// annoyance avoidance....
	Element *sumElem = AddElem(key);
	if (sumElem) {
		sumElem->label = valstr;
		return sumElem;
	} else switch(getKey(key)) {
	case TITLE:
		label = valstr;	// top level label is window title
		break;
	case PANELPOS:
		getPanelPos(valstr);
		break;
	case CANCEL:
		cancel = valstr[0] ? valstr : NULL;	// empty string suppresses button	
		break;
	case DONE:
		done = valstr[0] ? valstr : NULL;	// empty string suppresses button
		break;
	case DIV:
		divider = atof(valstr);
		break;
	case MIN:
		min = atoi(valstr);
		break;
	case MAX:
		max = atoi(valstr);
		break;
	default:
		return Element::AddPair(key, value);
	}
	return true;
}

bool PanelElem::getPanelPos(char *pstr) {
	char wstr[128];	// should be enough...
	char *token[5] ,*strp = wstr;
	int32 indx = 0;
	strncpy(wstr, pstr, 128);
	wstr[127] = 0;
	while (token[indx] = strtok(strp, " ")) {
		strp = NULL;
		if (indx < 4) indx++;	// strtok is WEIRD! -- must complete!
			
	}
	if (!indx) return false;
	int32 toki = 0;
	int32 val, x=100, y=100;
	char *endptr;
	if (toki < indx) {
		val = strtol(token[toki], &endptr, 10);
		if (endptr > (char *)token[toki]) x = val;
		toki++;
	}
	if (toki < indx) {
		val = strtol(token[toki], &endptr, 10);
		if (endptr > (char *)token[toki]) y = val;
	}
	panelPos = BPoint(x, y);
	return true;
}



Element *PanelElem::AddKey(unsigned char *key) {
	return AddElem(key);
}


Element * PanelElem::AddElem(unsigned char *key) {
	Element *subElem = NULL;
	KeyID id = getKey(key);
	switch (id) {
	case TEXTCONTROL:
		subElem = new TextCtrlElem(this);
	 	((TextCtrlElem *)subElem)->divider = divider >= 0 ? divider : width/2.0; 
		break;
	case MENU:
		subElem = new MenuElem(this);
		break;
	case LABEL:
		subElem = new LabelElem(this);
		break;
	case BUTTON:
		subElem = new ButtonElem(this);
		break;
	case CHECKBOX:
		subElem = new CheckBoxElem(this);
		break;
	case SLIDER:
		{
			SliderElem *sldrel = new SliderElem(this);
			subElem = sldrel;
		 	sldrel->min = min;
		 	sldrel->max = max;
		 	sldrel->init = (min + max)/2;
		}
		break;
	}
	if (subElem) {
		controls.AddItem(subElem);
		// defaults from parent:
		subElem->posmode = posmode;
		subElem->xoffset = xoffset;
		subElem->yoffset = yoffset;
		subElem->width = width;
		subElem->height = height;
	}
	return subElem;
}


TextCtrlElem::TextCtrlElem(Element *Parent=NULL)
 : initstr(NULL), Element(TEXTCTRL_ELEM, Parent){}

bool TextCtrlElem::AddPair(unsigned char *key, unsigned char *value) {
	char *valstr = (char *)value;	// annoyance avoidance....
	switch(getKey(key)) {
	case INIT:
		initstr = valstr;
		break;
	case DIV:
		divider = atof(valstr);
		break;
	default:
		return Element::AddPair(key, value);
	}
	return true;
}


MenuElem::MenuElem(Element *Parent=NULL) : Element(MENU_ELEM, Parent) {}

bool MenuElem::AddPair(unsigned char *key, unsigned char *value) {
	char *valstr = (char *)value;	// annoyance avoidance....
	switch(getKey(key)) {
	case INIT:
		marked = atoi(valstr) - 1;	// user range -> interface range
		break;
	default:
		return Element::AddPair(key, value);
	}
	return true;
}


LabelElem::LabelElem(Element *Parent=NULL) : Element(LABEL_ELEM, Parent) {}


ButtonElem::ButtonElem(Element *Parent=NULL) : state(false), close(false),
 Element(BUTTON_ELEM, Parent) {}

bool ButtonElem::AddPair(unsigned char *key, unsigned char *value) {
	switch(getKey(key)) {
	case DONE:
		close = true;	// value ignored;
		break;
	default:
		return Element::AddPair(key, value);
	}
	return true;
}


CheckBoxElem::CheckBoxElem(Element *Parent=NULL) : state(false), Element(CHECKBOX_ELEM, Parent) {}

bool CheckBoxElem::AddPair(unsigned char *key, unsigned char *value) {
	char *valstr = (char *)value;	// annoyance avoidance....
	switch(getKey(key)) {
	case INIT:
		if (strcmp(valstr, "on") == 0) state = true;
		else state = false;
		break;
	default:
		return Element::AddPair(key, value);
	}
	return true;
}


SliderElem::SliderElem(Element *Parent=NULL) : min(0), max(100), init(50), Element(SLIDER_ELEM, Parent) {}

bool SliderElem::AddPair(unsigned char *key, unsigned char *value) {
	char *valstr = (char *)value;	// annoyance avoidance....
	switch(getKey(key)) {
	case MIN:
		min = atoi(valstr);
		break;
	case MAX:
		max = atoi(valstr);
		break;
	case INIT:
		init = atoi(valstr);
		break;
	default:
		return Element::AddPair(key, value);
	}
	return true;
}


extern int buildPanel(PanelElem *spec);

void showTree(PanelElem *p) {
	buildPanel(p);
}


bool
parseYAML(const unsigned char *yamlstr, int32 yamlen) {
	int done = 0;

	yaml_parser_t parser;
	yaml_event_t event;
	Element *curr_elem = NULL;
	unsigned char *curr_key = NULL;
	PanelElem *root;

	/* Clear the objects. */

	memset(&parser, 0, sizeof(parser));
	memset(&event, 0, sizeof(event));

	/* Initialize the parser object. */

	if (!yaml_parser_initialize(&parser))
		goto parser_error;

	/* Set the parser parameters. */

	yaml_parser_set_input_string(&parser, yamlstr, yamlen);


	/* The main loop. */

	while (!done)
	{
		/* Get the next event. */

		if (!yaml_parser_parse(&parser, &event))
			goto parser_error;

		/* Handle the event, build display. */
		switch (event.type) {
		case YAML_STREAM_END_EVENT:
			 done = 1;
			break;
		case YAML_DOCUMENT_START_EVENT:
			if (curr_elem) {
				fprintf(stderr, "Can't handle more than one document\n");
				goto misc_error;
			}
			break;
		case YAML_MAPPING_START_EVENT:
			if (curr_elem && curr_elem->prevmode) {
				fprintf(stderr, "Can't handle more than one mapping level\n");
				goto misc_error;
			}
			if (!curr_elem) {
				curr_elem = root = new PanelElem;
				curr_elem->currmode = MAP_KEY;
			}
			else if (curr_elem->currmode == MAP_VALUE) {
 			   	Element *subElem = curr_elem->AddKey(curr_key);
 			   	curr_elem->currmode = MAP_KEY;
 			   	if (subElem) {
 				   	curr_elem = subElem;
	 			   	curr_elem->currmode = MAP_KEY;
 			   	}
			} else {
				curr_elem->prevmode = curr_elem->currmode;	// superfluous now?
				curr_elem->currmode = MAP_KEY;
			}
			break;
		case YAML_MAPPING_END_EVENT:
			curr_elem = curr_elem ? curr_elem->parent : NULL;
			break;
		case YAML_SEQUENCE_START_EVENT:
			if (!curr_elem || curr_elem->currmode != MAP_VALUE) {
				fprintf(stderr, "badly placed list!\n");
				goto misc_error;
			} else if (curr_elem->eltype == PANEL) {
	 		   	Element *subElem = curr_elem->AddKey(curr_key);
	 		   	curr_elem->currmode = MAP_KEY;
	 		   	if (subElem) {
	 			   	curr_elem = subElem;
	 		   	}
				curr_elem->currmode = SEQUENCE;
			} else if (curr_elem->eltype == MENU_ELEM && getKey(curr_key) == ITEMS) {
				curr_elem->currmode = SEQUENCE;
				curr_elem->parent->currmode = NO_POP;	// Temp lock to prevent creation
	 		}
			break;
		case YAML_SEQUENCE_END_EVENT:
			if (curr_elem->parent->currmode == MAP_KEY) {
				curr_elem = curr_elem->parent;
			} else {
				curr_elem->parent->currmode = MAP_KEY;	//restore from NO_POP
				curr_elem->currmode = MAP_KEY;	//restore from SEQUENCE
			}
			break;
		case YAML_SCALAR_EVENT:
			if (!curr_elem) {
				fprintf(stderr, "badly formatted string!\n");
				goto misc_error;
			}
			if (curr_elem->currmode == MAP_KEY) {
				curr_key = event.data.scalar.value;
 			   	curr_elem->currmode = MAP_VALUE;
			}
			else if (curr_elem->currmode == MAP_VALUE) {
 			   	curr_elem->AddPair(curr_key, event.data.scalar.value);
 			   	curr_elem->currmode = MAP_KEY;
			}
			else if (curr_elem->currmode == SEQUENCE) {
 			   	if (curr_elem->eltype != MENU_ELEM) break; // ignore bad list
				MenuElem *melem = (MenuElem *)curr_elem;
				melem->items.AddItem(event.data.scalar.value);
			}
			break;
		}

	}
	
	showTree(root);

	yaml_parser_delete(&parser);

	return true;

parser_error:

    /* Display a parser error message. */

    switch (parser.error)
    {
        case YAML_MEMORY_ERROR:
            fprintf(stderr, "Memory error: Not enough memory for parsing\n");
            break;

        case YAML_READER_ERROR:
            if (parser.problem_value != -1) {
                fprintf(stderr, "Reader error: %s: #%X at %d\n", parser.problem,
                        parser.problem_value, parser.problem_offset);
            }
            else {
                fprintf(stderr, "Reader error: %s at %d\n", parser.problem,
                        parser.problem_offset);
            }
            break;

        case YAML_SCANNER_ERROR:
            if (parser.context) {
                fprintf(stderr, "Scanner error: %s at line %d, column %d\n"
                        "%s at line %d, column %d\n", parser.context,
                        parser.context_mark.line+1, parser.context_mark.column+1,
                        parser.problem, parser.problem_mark.line+1,
                        parser.problem_mark.column+1);
            }
            else {
                fprintf(stderr, "Scanner error: %s at line %d, column %d\n",
                        parser.problem, parser.problem_mark.line+1,
                        parser.problem_mark.column+1);
            }
            break;

        case YAML_PARSER_ERROR:
            if (parser.context) {
                fprintf(stderr, "Parser error: %s at line %d, column %d\n"
                        "%s at line %d, column %d\n", parser.context,
                        parser.context_mark.line+1, parser.context_mark.column+1,
                        parser.problem, parser.problem_mark.line+1,
                        parser.problem_mark.column+1);
            }
            else {
                fprintf(stderr, "Parser error: %s at line %d, column %d\n",
                        parser.problem, parser.problem_mark.line+1,
                        parser.problem_mark.column+1);
            }
            break;

        default:
            /* Couldn't happen. */
            fprintf(stderr, "Internal error\n");
            break;
    }

misc_error:
    yaml_parser_delete(&parser);

    return false;

}

