
/*

  Project Sayanara
  Copyright (C) 2001 Mark Adams

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.
  
  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.
  
  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

*/

#include <iostream>
#include <string>
#include <list>
#include <stack>

#include "ssml.h"

SsmlTag *SsmlTag::child(string tag) {

  list<SsmlTag *>::iterator it = children->begin();
  
  while (it != children->end()) {
    if ((*it)->tag == tag)
      return (*it);
    it++;
  }
  
  return 0;

}

SsmlContent *SsmlStructure::child(string &name) {
  list<SsmlContent *>::iterator item = children.begin();
  while (item != children.end()) {
    SsmlContent *i = *item;
    if (i->type() == SsmlContent::STRUCTURE) {
      SsmlStructure *s = (SsmlStructure *)i;
      if (s->name == name)
	return s;
    }
    item++;
  }
  return 0;
}

string *SsmlStructure::stringContent() {
  if (children.size() != 1) return 0;
  SsmlContent *c = *(children.begin());
  if (c->type() != SsmlStructure::STRING) return 0;
  SsmlString *s = (SsmlString *) c;
  return &(s->content);
}

string *SsmlStructure::childStringContent(char *name) { 
  string s = name; 
  return childStringContent(s);
}

string *SsmlStructure::childStringContent(string &name) {
  SsmlContent *c = child(name);
  if (c == 0) return 0;
  if (c->type() != SsmlContent::STRUCTURE) return 0;
  SsmlStructure *s = (SsmlStructure *) c;
  return s->stringContent();
}

void SsmlStructure::destroy() {
  list<SsmlContent *>::iterator item = children.begin();
  while (item != children.end()) {
    (*item)->destroy();
    item++;
  }
}

void SsmlStructure::dump(int level) {
  for(int i = 0; i < level * 2; i++) cout << "  ";
  cout << "struct " << name << endl;
  list<SsmlContent *>::iterator item = children.begin();
  while (item != children.end()) {
    (*item)->dump(level + 1);
    item++;
  }
}

void SsmlTokeniser::next() {

  while (input->peek() == '\n' || input->peek() == '\r' || 
	 input->peek() == ' ' || input->peek() == '\t')
    input->get();

  if (input->peek() == EOF) {
    tType = END_OF_FILE;
    return;
  }
    
  if (input->peek() == '<')
    nextTag();
  else
    nextString();
  
}

void SsmlTokeniser::nextTag() {

  input->get();			// Throw away the '<'
  if (input->peek() == '/')
    nextCloseTag();
  else
    nextOpenTag();
  
}

void SsmlTokeniser::nextOpenTag() {

  tType = OPEN_TAG;
  posn = 0;

  while(input->peek() != EOF && input->peek() != '>')
    tok[posn++] = input->get();

  tok[posn] = 0;
  
  if (input->peek() == '>')
    input->get();

}

void SsmlTokeniser::nextCloseTag() {

  input->get();			// Throw away the '/'
  tType = CLOSE_TAG;
  posn = 0;

  while(input->peek() != EOF && input->peek() != '>')
    tok[posn++] = input->get();

  tok[posn] = 0;
  
  if (input->peek() == '>')
    input->get();

}

void SsmlTokeniser::nextString() {

  tType = STRING;
  posn = 0;

  while (input->peek() != EOF && input->peek() != '<') {
    if (input->peek() == '&') {
      input->get();
      string s;
      while (input->peek() != ';' && input->peek() != EOF && s.length() < 5)
	s += input->get();
      if (input->peek() == ';') input->get();
      if (s == "amp") tok[posn++] = '&';
      else if (s == "lt") tok[posn++] = '<';
      else if (s == "gt") tok[posn++] = '>';
      else if (s == "nl") tok[posn++] = '\n';
      else if (s == "quot") tok[posn++] = '"';
    } else if (input->peek() == '%') {
      int val = 0;
      input->get();
      for(int i = 0; i < 2; i++) {
	val *= 16;
	if (input->peek() >= '0' && input->peek() <= '9')
	  val += input->get() - '0';
	else
	  val += input->get() - 'A' + 10;
      }
      tok[posn++] = val;
    } else
      tok[posn++] = input->get();
  }

  while(input->peek() != EOF && input->peek() != '<')
    tok[posn++] = input->get();

  while (posn > 0 && 
	 (tok[posn-1] == '\n' || tok[posn-1] == '\r' || 
	  tok[posn-1] == ' ' || tok[posn-1] == '\t'))
    posn--;

  tok[posn] = 0;

  

}

SsmlTokeniser::TokenType SsmlTokeniser::tokenType() {
  return tType;
}

string &SsmlTokeniser::token() {
  stok = tok;
  return stok;
}

SsmlContent *SsmlParser::parse() {

  position = spec;
  
  content = new SsmlStructure();
  current = content;

  do {
    tokeniser->next();
    if (tokeniser->tokenType() != SsmlTokeniser::END_OF_FILE)
      parseToken();
  } while (tokeniser->tokenType() != SsmlTokeniser::END_OF_FILE);

  return content;

}

void SsmlParser::parseToken() {

  SsmlTag *tag;

  switch(tokeniser->tokenType()) {

  case SsmlTokeniser::STRING:
    if (position->hasText) {
      SsmlString *here = new SsmlString(tokeniser->token());
      SsmlStructure *parent = (SsmlStructure *) current;
      parent->children.insert(parent->children.end(), here);
    } else {
      string e = "Tag " + position->tag + " doesn't expect string content.";
      error(e);
    }
    break;

  case SsmlTokeniser::OPEN_TAG:
    tag = position->child(tokeniser->token());
    if (tag == 0) {
      string e = "Tag " + tokeniser->token() + " not expected.";
      error(e);
    } else {
      tagStack.push(position);
      position = tag;

      SsmlStructure *here = new SsmlStructure(tokeniser->token());

      contentStack.push(current);

      SsmlStructure *parent = (SsmlStructure *) current;
      parent->children.insert(parent->children.end(), here);
      current = here;

    }
    break;
  
  case SsmlTokeniser::CLOSE_TAG:
    if (position->tag != tokeniser->token()) {
      string e = "Tag " + tokeniser->token() + " does not close me.";
      error(e);
    } else {
      position = tagStack.top();
      tagStack.pop();
      current = contentStack.top();
      contentStack.pop();
    }
    break;

  }
}

void SsmlParser::error(string &s) {
  cerr << "AgendaParser error: " << s << endl;
  exit(1);
}
