
/*

  Project Sayanara
  Copyright (C) 2001 Mark Adams

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.
  
  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.
  
  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

*/

#include <iostream>
#include <string>
#include <list>
#include <stack>

class SsmlTokeniser {
public:
  enum TokenType { OPEN_TAG, CLOSE_TAG, STRING, END_OF_FILE };
  SsmlTokeniser(istream &input) { this->input = &input; }
  void next();
  TokenType tokenType();
  string &token();
private:
  static const int maxTokenLength = 1024;
  void nextTag();
  void nextString();
  void nextOpenTag();
  void nextCloseTag();
  istream *input;
  char tok[maxTokenLength];
  string stok;
  TokenType tType;
  int posn;
};

class SsmlTag {
public:
  string tag;
  bool hasText;
  list<SsmlTag *> *children;

  SsmlTag(string tag, bool hasText, list<SsmlTag *> *children) {
    this->tag = tag; this->hasText = hasText;
    this->children = children;
  }

  SsmlTag(string tag, bool hasText) {
    this->tag = tag; this->hasText = hasText;
    children = new list<SsmlTag *>;
  }

  void addChild(SsmlTag *tag) {
    children->insert(children->end(), tag);
  }

  SsmlTag *child(string tag);

};

class SsmlContent {
public:
  virtual void destroy() = 0;
  virtual void dump(int level) = 0;
  void dump() { dump(0); }
  enum Type { STRING, STRUCTURE };
  virtual Type type() = 0;
};

class SsmlString : public SsmlContent {
public:
  string content;
  void destroy() {}
  SsmlString(string &s) { content = s; }
  SsmlString() { }
  void dump(int level) {
    for(int i = 0; i < level * 2; i++) cout << "  ";
    cout << "string " << content << endl;
  }
  virtual Type type() { return STRING; }
};

class SsmlStructure : public SsmlContent {
public:
  string name;
  list<SsmlContent *> children;
  SsmlStructure(string &s) { name = s; }
  SsmlStructure() { }
  SsmlContent *child(string &name);
  string *stringContent();
  string *childStringContent(char *name);
  string *childStringContent(string &name);
  void destroy();
  void dump(int level);
  virtual Type type() { return STRUCTURE; }
};

class SsmlParser {
public:
  SsmlParser(SsmlTokeniser &x) { tokeniser = &x; }
  SsmlContent *parse();
private:
  SsmlContent *content;
  SsmlContent *current;
  SsmlTag *position;
  void parseToken();
  void error(string &x);
  void error(char *x) { string e(x); error(e); }
  stack<SsmlTag *> tagStack;
  stack<SsmlContent *> contentStack;
protected:
  SsmlTokeniser *tokeniser;
  SsmlTag *spec;
};


class AgendaParser : public SsmlParser {
public:
  AgendaParser(SsmlTokeniser &x) : SsmlParser(x) {
    
    SsmlTag *agenda = new SsmlTag("agenda", false);

    SsmlTag *appt = new SsmlTag("appt", false);
    appt->addChild(new SsmlTag("start", true));
    appt->addChild(new SsmlTag("end", true));
    appt->addChild(new SsmlTag("text", true));

    SsmlTag *event = new SsmlTag("event", false);
    event->addChild(new SsmlTag("start", true));
    event->addChild(new SsmlTag("end", true));
    event->addChild(new SsmlTag("text", true));

    SsmlTag *todo = new SsmlTag("todo", false);
    todo->addChild(new SsmlTag("crossedout", true));
    todo->addChild(new SsmlTag("due", true));
    todo->addChild(new SsmlTag("text", true));
    todo->addChild(new SsmlTag("priority", true));

    SsmlTag *anniversary = new SsmlTag("anniversary", false);
    anniversary->addChild(new SsmlTag("text", true));
    anniversary->addChild(new SsmlTag("start", true));

    SsmlTag *daynote = new SsmlTag("daynote", false);
    daynote->addChild(new SsmlTag("display", true));
    daynote->addChild(new SsmlTag("text", true));

    agenda->addChild(appt);
    agenda->addChild(event);
    agenda->addChild(todo);
    agenda->addChild(anniversary);
    agenda->addChild(daynote);

    spec = new SsmlTag("outside", false);
    spec->addChild(agenda);

  }
};

class ContactsParser : public SsmlParser {
public:
  ContactsParser(SsmlTokeniser &x) : SsmlParser(x) {
    
    SsmlTag *contacts = new SsmlTag("contacts", false);

    SsmlTag *contact = new SsmlTag("contact", false);

    SsmlTag *field = new SsmlTag("field", false);
    field->addChild(new SsmlTag("label", true));
    field->addChild(new SsmlTag("type", true));
    field->addChild(new SsmlTag("textvalue", true));
    field->addChild(new SsmlTag("datevalue", true));

    contact->addChild(field);

    contacts->addChild(contact);

    spec = new SsmlTag("outside", false);
    spec->addChild(contacts);

  }
};
