// Sayanara (EPOC) Server access from BeOS
// Copyright Pete Goodeve 2002
// This is free software, and may be distributed without restraint.
// Absolutely NO warranty is expressed or implied.

#include <stdio.h>
#include <unistd.h>
#include <string.h>
#include <Application.h>
#include <String.h>
#include <getopt.h>

#include "psionconn.h"

#define CONTACTS_SERVER "SYS$SAYANCNT"
#define AGENDA_SERVER "SYS$SAYANAGN"
#define CONTACTS_FILE "C:\\System\\Data\\Contacts.cdb"
#define AGENDA_FILE "C:\\Documents\\Agenda"

// Some OSs never learn... (:-/)
void slash2back(char *str) {
	while (*str) {
		if (*str == '/') *str = '\\';
		str++;
	}
}

class SayaApplication : public BApplication 
{
	bool verbose;
	enum {CONTACTS, AGENDA} mode;
	BString filename, output;
	char *servername;
public:
	SayaApplication() : BApplication("application/x-PsiONtrack.Sayanara"),
	 verbose(false), mode(CONTACTS), filename(NULL), output(NULL),
	servername(CONTACTS_SERVER) {}
	
	void ArgvReceived(int32 argc, char **argv) {
		int optc;
		int digit_optind = 0;
		struct option options[] = {
			{"help", 0, 0, 'h'},
			{"agenda", 0, 0, 'a'},
			{"contacts", 0, 0, 'c'},
			{"file", 1, 0, 'f'},
			{"output", 1, 0, 'o'},
			{"verbose", 0, 0, 'v'},
			{"version", 0, 0, 'V'},
			{0, 0, 0, 0}
		};

		while (true) {
			int this_option_optind = optind ? optind : 1;
			int option_index = 0;

			optc = getopt_long(argc, argv, "hacf:o:vV",
					options, &option_index);
			if (optc == -1) break;

			switch (optc) {
			case 'h':
				fprintf(stderr, "\nUsage: sayanara <OPTIONS>\n\n" \
				"    Options:\n" \
				"      -h, --help           Print this message and exit.\n" \
				"      -V, --version        Print version and exit.\n" \
				"      -a, --agenda         Read Agenda file on EPOC device.\n" \
				"      -c, --contacts       Read Contacts file on EPOC device (default).\n" \
				"      -f, --file=<file>    Specify (non-default) file on EPOC device.\n" \
				"      -o, --output=<file>  Specify output file.\n" \
				"    (default is to read Contacts to standard output)\n" \
//				"      -v, --verbose        Increase verbosity.\n" 
				);
				Quit();
				break;
			case 'f':
				slash2back(optarg);	// a courtesy to sane OSs
				filename = optarg;
				break;
			case 'a':
				mode = AGENDA;
				break;
			case 'c':
				mode = CONTACTS;
				break;
			case 'o':
				output = optarg;
				break;
			case 'v':
				verbose = true;
				break;
			case 'V':
				fprintf(stderr, "Version 0.0.1\n");
				Quit();
				return;
			}
		}
		if (optind < argc) {
			fprintf(stderr, "Unknown argument: %s\n", argv[optind]);
			Quit();
			return;
		}
	}	// end of ArgvReceived
	
	void ReadyToRun(void) {
		if (mode == AGENDA) servername = AGENDA_SERVER;	// otherwise default CONTACTS
		if (!filename.Length()) filename = mode == AGENDA ?
									AGENDA_FILE : CONTACTS_FILE;
		
		PsionConnector cnct(servername);	// will disconnect itself at exit
		FILE *outstream = stdout;
		
		if (cnct.connected()) {
			if (output.Length()) {
				outstream = fopen(output.String(), "w");
				if (!outstream) {
					fprintf(stderr, "Couldn't open %s for output\n", output.String());
					PostMessage(B_QUIT_REQUESTED);
					return;
				}
			} 
			char cmdpkt[128];	
			cmdpkt[0] = 2; // set-up command ('fetch')
			strncpy(cmdpkt+1, filename.String(), 126);
			cmdpkt[127] = 0; //being paranoid...
//			printf("invoking file '%s', pkt len %d\n", cmdpkt+1, strlen(cmdpkt+1));			
			int len = cnct.transaction(cmdpkt, strlen(cmdpkt+1)+1); // null terminator NOT included!
			int8 *resp_data = (int8 *)cnct.response();
			if (len <= 0 || resp_data[0] != 0) {	//  don'tseem to be in touch
				fprintf(stderr, "SAYANARA server failure!\n");
				PostMessage(B_QUIT_REQUESTED);
				return;
			}
			
			fprintf(stderr, "Fetching Data...\n");
			int8 readcode = 1;	// 'read' command
			do {
				len = cnct.transaction(&readcode, 1); // request some file data
				resp_data = (int8 *)cnct.response();
				if (len > 1 && resp_data[0] == 0) 
					fprintf(outstream, "%.*s", len-1, resp_data+1);
			} while (len > 1 && resp_data[0] == 0);
			if (outstream != stdout) fclose(outstream);
		}
		PostMessage(B_QUIT_REQUESTED);
	}
};



int main(void)
{	
	SayaApplication	myApplication;
	myApplication.Run();
	return(0);
}

